//___________________________________
//                                   \
// Sound Engine.  version 1.3		# $Id: acusthica.js 60 2007-09-09 06:44:43Z nilton $
//___________________________________/

/*
 Preloads sounds, handles pauses (Collaborates with Ethernum)
*/


/**
 * Create an AcusthicaEngine Object
 * @param {Object} ethernumobject An Ethernum Object to enable pause BGM during a pause. 
 * 
 * example:
 * 	var Pause=new EthernumEngine(<parameters>);
 * 	var MUSIC=new AcusthicaEngine(Pause);
 * or:
 * 	var MUSIC=new AcusthicaEngine();
 * 	Pause.SetAcusthica(MUSIC);
 * TODO:
 * Add random pitched 'speech' sounds and panfromto sound
*/
function AcusthicaEngine(ethernumobject) {
	if(this instanceof AcusthicaEngine == false) 
		return new AcusthicaEngine(ethernumobject);

	this.bgm;
	this.bgmfilename;
	this.bgmstream = true;
	this.bgmloop = true;
	this.bgmvolume = 255;
	this.Sounds = new Array();
	if(ethernumobject)
		ethernumobject.music = this;
	this.sfxdefaultvolume =255;
	this.pauseSfxToo = false;
}

/**
 * Plays a background music (BGM) file
 * @param {string} filename Name of a sound file in ./sounds/
 * @param {Boolean} stream Is the soundfile streamed? 
 * @param {Boolean} loop If the BGM is looped, or just played once.
 * example:
 * 	var MUSIC=new AcusthicaEngine();
 * 	MUSIC.bgmvolume=200; //Background music a little softer
 * 	MUSIC.play("melody.mp3");
 */
AcusthicaEngine.prototype.play= function(filename,stream,loop) {
	this.bgmfilename = filename||this.bgmfilename;
	if(typeof stream == "boolean")
		this.bgmstream = stream? true : false;
	if(typeof loop == "boolean")
		this.bgmloop= loop? true : false;
	this.bgm = LoadSound(this.bgmfilename,this.bgmstream);
	this.bgm.setVolume(this.bgmvolume);
	this.bgm.play(this.bgmloop);
}

/**
 * Tells us if the BGM is playing
 * @returns True is the Background music is playing, false if not
 * @type Boolean
 */
AcusthicaEngine.prototype.isPlaying= function() {
	if(this.bgm && this.bgm.isPlaying())
		return true;
	return false;
}

/**
 * pauses BGM music
 * @param {Boolean} pauseSfxToo Also pauses the {@link AcusthicaEngine#sfx} Sounds
 * @returns True if there was a BGM we could pause.
 */
AcusthicaEngine.prototype.pause= function(pauseSfxToo) {
	pauseSfxToo = pauseSfxToo===undefined? this.pauseSfxToo : pauseSfxToo;
	if(pauseSfxToo){
		for(fn in this.Sounds){
			if(this.Sounds[fn].isPlaying())
				this.Sounds[fn].pause();
		}
	}
	if(this.bgm && this.bgm.isPlaying()){
		this.bgm.pause();
		this.bgm.setVolume(this.bgmvolume);
		return true;
	}
	return false;
}

/**
 * unpauses BGM music
 * @param {Boolean} pauseSfxToo Also pauses the {@link AcusthicaEngine#sfx} Sounds
 * @returns True if there was a BGM we could unpause.
 */
AcusthicaEngine.prototype.unpause= function(pauseSfxToo) {
	pauseSfxToo = pauseSfxToo===undefined? this.pauseSfxToo : pauseSfxToo;
	if(pauseSfxToo){
		for(fn in this.Sounds){
			if(this.Sounds[fn].isSeekable() && this.Sounds[fn].getPosition()>0)
				this.Sounds[fn].play();
		}
	}
	if(this.bgm && !this.bgm.isPlaying()){
		this.bgm.play(this.bgmloop);
		this.bgm.setVolume(this.bgmvolume);
		return true;
	}
	return false;
}

/**
 * stops BGM music
 * @returns True if there was a BGM we could stop.
 */
AcusthicaEngine.prototype.stop= function(){
	if(this.bgm && this.bgm.isPlaying()){
		this.bgm.stop();
		this.bgm = undefined;
		return true;
	}
	return false;
}

/**
 * Fades a sound from a volume to another volume
 * @param {string} sound The filename of the sfx, is 0/undefined/false for BGM
 * @param {integer} from Starting volume (0..255)
 * @param {integer} to Ending volume (0..255)
 * @param {integer} step Volume stepping in frames per step
 * @param {Boolean} pauseafter Pause the sound after volume has reached 'to'
 * @param {string} playafter The filename of the next BGM/sfx to play
 * @param {Boolean} stream Is the BGM streamed OR the volume of the sfx. False for defaults.
 * TODO: this code looks too much like python, need to reformat it.
 */
AcusthicaEngine.prototype.fadeSound=function(sound,from,to,step,pauseafter,playafter,stream) {
	step = step||1;
	if(Math.abs(from-to)<step)
		step=1;

	if(from>to)
		from-=step;
	else
		from+=step;

	if(sound)
		this.Sounds[sound].setVolume(from);
	else
		this.bgm.setVolume(from);

	if(from != to){
		SetDelayScript(1, this.fadeSound(sound,from,to,step,pauseafter,playafter,stream));
		return;
	} else {
		if(pauseafter){
			if(sound)
				this.Sounds[sound].pause();
			else
				this.bgm.pause();
		}else{
			if(sound)
				this.Sounds[sound].stop();
			else
				this.bgm.stop();
		}
		if(playafter){
			if(sound)
				this.sfx(filename,stream); //Volume==stream
			else
				this.play(playafter,stream);
		}
	}
}

/**
 * Abruptly stop the BGM music if it was playing, and start a new song
 * @param {string} filename The name of the song file to play.
 * @param {Boolean} stream True if the song must be streamed.
 * @param {Boolean} loop If the BGM is looped, or just played once. 
 */
function ChangeMusic(filename,stream,loop) {
	if(this.bgm)
		this.bgm.stop();
	this.play(filename,stream,loop);
}
 
/**
 * Fades the music out and maybe play another song
 * @param {string} filename The name of the next sound file to play. Optional
 * @param {Boolean} stream True if the next sound must be streamed.
 */
function FadeMusic(filename,stream) {
	if(filename)
		this.fadeSound(0,this.bgmvolume,0,16,0,filename,stream);
	else
		this.fadeSound(0,this.bgmvolume,0,16,1);
}

/**
 * Cache a short sound file.
 * @param {string} filename The name of the sound file to cache
 * @param {integer} defaultvolume Optional number between 0 and 255 (will default to .sfxdefaultvolume (255, if you dont chage it))
 * @param {float} defaultpitch Optional number between 0.5 and 2.0 . Defaults to 1.
 */
AcusthicaEngine.prototype.sfxLoad = function(filename,defaultvolume,defaultpitch) {
	if(!this.Sounds[filename])
		this.Sounds[filename] = LoadSound(filename);
	this.Sounds[filename].setPan(0);
	if(typeof defaultvolume != 'number')
		defaultvolume = this.sfxdefaultvolume;
	this.Sounds[filename].setVolume(defaultvolume);
	this.Sounds[filename].defaultvolume = defaultvolume;
	this.Sounds[filename].defaultpitch = defaultpitch||1;
	this.Sounds[filename].setPitch(defaultpitch||1);
}

/**
 * Play a cached sound file. Not looped.
 * @param {string} filename The name of the sound file
 * @param {integer} defaultvolume Number between 1 and 255 (if 0, undefined it will default to .sfxdefaultvolume (255, if you dont chage it))
 * @param {integer} pan Number between -255 (left) and 255 (right), will default to 0 (centered).
 * @param {float} pitch Number between 0.5 and 2.0 or an octave string, see {@link octave2pitch}. Optional
 */
AcusthicaEngine.prototype.sfx = function(filename,volume,pan,pitch) {
	if(!this.Sounds[filename])
		this.sfxLoad(filename);
	this.Sounds[filename].stop();
	this.Sounds[filename].setPan(pan||0);
	this.Sounds[filename].setVolume(volume||this.Sounds[filename].defaultvolume);
	if(typeof pitch == 'string')
			this.Sounds[filename].setPitch(this.o2p[pitch]);
	else
		this.Sounds[filename].setPitch(pitch||this.Sounds[filename].defaultpitch);
	this.Sounds[filename].play(false);
}

/**
 * Returns the Sphere pitch of an octave note string
 * @param {string} note The note to convert to a pitch.
 * c d e f g a b and C D E F G A B C2 with corresponding flats '#'
 * @returns a pitch between 0.5 and 2.0
 * The fractions are weird, because the lower octave is beween 0.5 and 1.0 (a range of 0.5) and
 * the upper octave is between 1.0 and 2.0 (a range of 1.0). 
 * TODO: Maybe enhance with microtonal values: octatonic, lydian, dorian etc. And the original UT-RE-MI scale
 */
AcusthicaEngine.prototype.octave2pitch = function(note){ return this.o2p[note]; }
AcusthicaEngine.prototype.o2p = {
        'c': 0.5 + 0/24, 'c#': 0.5 + 1/24,
        'd': 0.5 + 2/24, 'd#': 0.5 + 3/24,
        'e': 0.5 + 4/24,
        'f': 0.5 + 5/24, 'f#': 0.5 + 6/24,
        'g': 0.5 + 7/24, 'g#': 0.5 + 8/24,
        'a': 0.5 + 9/24, 'a#': 0.5 + 10/24,
        'b': 0.5 + 11/24,

        'C': 1.0 + 0/12, 'C#': 1.0 + 1/12,
        'D': 1.0 + 2/12, 'D#': 1.0 + 3/12,
        'E': 1.0 + 4/12,
        'F': 1.0 + 5/12, 'F#': 1.0 + 6/12,
        'G': 1.0 + 7/12, 'G#': 1.0 + 8/12,
        'A': 1.0 + 9/12, 'A#': 1.0 + 10/12,
        'B': 1.0 + 11/12,
        'C2': 1.0 + 12/12,
};


/**
 * Plays a sound, but will change the panning according to the (xx,yy) coordinates
 * @param {string} filename The name of the sound file
 * @param {number} xx Map coordinates where the sound comes from.
 * @param {number} yy Map coordinates where the sound comes from.
 */
AcusthicaEngine.prototype.sfxSpacial = function(filename,xx,yy) {
	var dx = GetCameraX()-xx;
	var d2 = ( dx*dx + (GetCameraY()-yy)*(GetCameraY()-yy) ) ;
	var volume = 255;
	if(d2>(64*64)) 
		volume= Math.floor(255*256*16/d2);
	if(volume==0)
		return;
	var pan = Math.floor(dx*0.5);
	if(pan>255)
		pan=255;
	if(pan<-255)
		pan=-255;
	if(!this.Sounds[filename])
		this.Sounds[filename] = LoadSound(filename);
	this.Sounds[filename].stop();
	this.Sounds[filename].setVolume(volume);
	this.Sounds[filename].setPan(pan);
	this.Sounds[filename].play(false);
}

/**
 * Returns how many sounds are currently cached
 * @returns The number of loaded sounds
 */
AcusthicaEngine.prototype.loadedSounds=function() { return this.Sounds.length; }

/**
 * Clears the sounds cache
 */
AcusthicaEngine.prototype.clearSounds=function() { this.Sounds=[]; }


